"use client"

import { Dialog, DialogContent } from "@/components/ui/dialog"
import { Sidebar, SidebarContent, SidebarGroup, SidebarGroupContent, SidebarGroupLabel, SidebarMenu, SidebarMenuButton, SidebarMenuItem, SidebarProvider } from "@/components/ui/sidebar"
import { useApplicationWorkflow } from "@/hooks/use-application-workflow"
import { TransformedAgenciesData } from "@/types/transformers"
import { ComponentWithClassName } from "@/types/utils"
import { WorkflowOption } from "@/types/workflow"
import { BadgeInfo, Download, Radio, UploadCloud } from "lucide-react"
import { StatusRevalidateBannerModalSidebar } from "./status.revalidate-banner-modal-sidebar"
import { WorkflowDialogDownload } from "./workflow.dialog-download"
import { WorkflowDialogStatus } from "./workflow.dialog-status"
import { WorkflowDialogUpload } from "./workflow.dialog-upload"
import { WorkflowDialogView } from "./workflow.dialog-view"

export function WorkflowDialog(props: ComponentWithClassName<{ data: NonNullable<TransformedAgenciesData> }>) {
  const {
    resetWorkflow,
    setWorkflow,
    workflow,
  } = useApplicationWorkflow()

  if (!workflow) {
    return
  }

  if (
    workflow.__tag !== "UPLOAD_AGENCY_FEED" &&
    workflow.__tag !== "STATUS_AGENCY_FEED" &&
    workflow.__tag !== "VIEW_AGENCY_FEED" &&
    workflow.__tag !== "DOWNLOAD_AGENCY_FEED"
  ) {
    return;
  }

  const potential = props.data.find((d) => d.gtfsId === workflow.id)
  if (!potential) {
    return
  }

  const content = INTERNAL__WorkflowDialogContent(workflow, props, resetWorkflow)

  return content && (
    <Dialog open defaultOpen onOpenChange={resetWorkflow} key={workflow.id}>
      <DialogContent className="overflow-hidden p-0 md:max-h-[500px] md:max-w-[700px] lg:max-w-[800px]">
        <SidebarProvider className="items-start min-h-[calc(100svh-12rem)] md:min-h-[32rem]">
          <Sidebar collapsible="none" className="hidden md:flex md:w-64 md:flex-col">
            <SidebarContent className="flex flex-1 flex-col overflow-hidden">
              <StatusRevalidateBannerModalSidebar agency={potential.gtfsId} />

              <SidebarGroup>
                <SidebarGroupLabel>Agencies</SidebarGroupLabel>
                <SidebarGroupContent>
                  <SidebarMenu>
                    {potential.access !== "ALL" && (
                      <SidebarMenuItem>
                        <SidebarMenuButton
                          isActive={workflow.__tag === "VIEW_AGENCY_FEED"}
                          onClick={() => {
                            if (!('id' in workflow)) {
                              return;
                            }

                            setWorkflow({
                              __tag: "VIEW_AGENCY_FEED",
                              id: workflow.id
                            })
                          }}
                        >
                          <BadgeInfo className="size-3" />
                          <span>Details</span>
                        </SidebarMenuButton>
                      </SidebarMenuItem>
                    )}

                    {potential.access === "ALL" && (
                      <>
                        <SidebarMenuItem>
                          <SidebarMenuButton
                            isActive={workflow.__tag === "VIEW_AGENCY_FEED"}
                            onClick={() => {
                              if (!('id' in workflow)) {
                                return;
                              }

                              setWorkflow({
                                __tag: "VIEW_AGENCY_FEED",
                                id: workflow.id
                              })
                            }}
                          >
                            <BadgeInfo className="size-3" />
                            <span>Details</span>
                          </SidebarMenuButton>
                        </SidebarMenuItem>
                        <SidebarMenuItem>
                          <SidebarMenuButton
                            isActive={workflow.__tag === "STATUS_AGENCY_FEED"}
                            onClick={() => {
                              if (!('id' in workflow)) {
                                return;
                              }

                              setWorkflow({
                                __tag: "STATUS_AGENCY_FEED",
                                id: workflow.id
                              })
                            }}
                          >
                            <Radio className="size-3" />
                            <span>Feed status</span>
                          </SidebarMenuButton>
                        </SidebarMenuItem>
                        <SidebarMenuItem>
                          <SidebarMenuButton
                            isActive={workflow.__tag === "UPLOAD_AGENCY_FEED"}
                            onClick={() => {
                              if (!('id' in workflow)) {
                                return;
                              }

                              setWorkflow({
                                __tag: "UPLOAD_AGENCY_FEED",
                                id: workflow.id
                              })
                            }}
                          >
                            <UploadCloud className="size-3" />
                            <span>Upload new feed</span>
                          </SidebarMenuButton>
                        </SidebarMenuItem>
                        <SidebarMenuItem>
                          <SidebarMenuButton
                            isActive={workflow.__tag === "DOWNLOAD_AGENCY_FEED"}
                            onClick={() => {
                              if (!('id' in workflow)) {
                                return;
                              }

                              setWorkflow({
                                __tag: "DOWNLOAD_AGENCY_FEED",
                                id: workflow.id
                              })
                            }}
                          >
                            <Download className="size-3" />
                            <span>Download feed</span>
                          </SidebarMenuButton>
                        </SidebarMenuItem>
                      </>
                    )}
                  </SidebarMenu>
                </SidebarGroupContent>
              </SidebarGroup>
            </SidebarContent>
          </Sidebar>

          <div className="flex flex-col gap-4 p-6 w-full h-full">
            {content}
          </div>
        </SidebarProvider>
      </DialogContent>
    </Dialog>
  )
}

function INTERNAL__WorkflowDialogContent(workflow: NonNullable<WorkflowOption>, { data, className }: ComponentWithClassName<{ data: NonNullable<TransformedAgenciesData> }>, _setClose: () => void) {
  if (
    workflow.__tag !== "UPLOAD_AGENCY_FEED" &&
    workflow.__tag !== "STATUS_AGENCY_FEED" &&
    workflow.__tag !== "VIEW_AGENCY_FEED" &&
    workflow.__tag !== "DOWNLOAD_AGENCY_FEED"
  ) {
    return null;
  }

  const potential = data.find((d) => d.gtfsId === workflow.id)
  if (!potential) {
    return null;
  }

  if (workflow.__tag === "UPLOAD_AGENCY_FEED") {
    return <WorkflowDialogUpload data={potential} className={className} />
  }

  if (workflow.__tag === "VIEW_AGENCY_FEED") {
    return <WorkflowDialogView data={potential} className={className} />
  }

  if (workflow.__tag === "STATUS_AGENCY_FEED") {
    return <WorkflowDialogStatus data={potential} className={className} />
  }

  if (workflow.__tag === "DOWNLOAD_AGENCY_FEED") {
    return <WorkflowDialogDownload data={potential} className={className} />
  }


  return null
}

